<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Plan;
use App\Models\Subscription;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Validator;

class PlanController extends Controller
{

    public function index()
    {
        $plan = Plan::with('planFeature')->select('id', 'name', 'price', 'billing_cycle')
            ->orderBy('created_at', 'asc')
            ->get();

        return response()->json([
            'status' => 'valid',
            'message' => 'Plan fetched successfully',
            'data' => $plan
        ]);
    }


    public function subscribeNow(Request $request)
    {
        $user = auth()->user();

        $validator = Validator::make($request->all(), [
            'course_id' => 'required|integer',
            'plan_id' => 'required|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'invalid',
                'message' => $validator->errors()->first()
            ]);
        }

        // Check for existing subscription
        $existingSubscription = Subscription::where('user_id', $user->id)
            ->where('course_id', $request->course_id)
            ->where('plan_id', $request->plan_id)
            ->first();

        if ($existingSubscription) {
            return response()->json([
                'status' => 'exists',
                'message' => 'This plan is already subscribed.'
            ]);
        }

        // Get plan and calculate end date based on billing cycle
        $plan = Plan::find($request->plan_id);

        if (!$plan) {
            return response()->json([
                'status' => 'invalid',
                'message' => 'Invalid plan selected.'
            ]);
        }

        $startDate = Carbon::now();
        $endDate = match ($plan->billing_cycle) {
            'month' => $startDate->copy()->addMonth(),
            'year' => $startDate->copy()->addYear(),
            default => $startDate, // fallback, or you can throw error
        };

        $subscription = Subscription::create([
            'user_id' => $user->id,
            'course_id' => $request->course_id,
            'plan_id' => $request->plan_id,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'status' => 'active', // assuming you want to default to 'active'
        ]);

        return response()->json([
            'status' => 'valid',
            'message' => 'Subscribed successfully.',
            'data' => $subscription
        ]);
    }
    

    // public function subscription()
    // {
    //     $user = auth()->user();

    //     // Get all subscriptions for the user with related course and plan
    //     $subscriptions = Subscription::with(['course', 'plan'])
    //         ->where('user_id', $user->id)
    //         ->get()
    //         ->map(function ($sub) {
    //             return [
    //                 'course_image' =>$sub->course ? asset('uploads/courses/' . $sub->course->image) : null,
    //                 'course_name' => $sub->course->name ?? null,
    //                 'plan_name' => $sub->plan->name ?? null,
    //                 'end_date' => Carbon::parse($sub->end_date)->format('M d, Y'),
    //                 'subscriptions_data' => $sub,
    //             ];
    //         });

    //     return response()->json([
    //         'status' => 'valid',
    //         'message' => 'Subscriptions fetched successfully.',
    //         'data' => $subscriptions
    //     ]);
    // }
    
    public function subscription()
    {
        $user = auth()->user();
    
        // Get all subscriptions with related course and plan
        $subscriptions = Subscription::with(['course', 'plan'])
            ->where('user_id', $user->id)
            ->get();
    
        // Check if any subscription has no course
        $hasNullCourse = $subscriptions->contains(function ($sub) {
            return $sub->course === null;
        });
    
        if ($hasNullCourse) {
            // If any subscription has no course, return no data or an empty array
            return response()->json([
                'status' => 'invalid',
                'message' => 'Some subscriptions do not have associated courses.',
                'data' => []
            ]);
        }
    
        // Otherwise, map and return subscriptions as usual
        $subscriptionsData = $subscriptions->map(function ($sub) {
            return [
                'course_image' => asset('uploads/courses/' . $sub->course->image),
                'course_name' => $sub->course->name,
                'plan_name' => $sub->plan->name ?? null,
                'end_date' => Carbon::parse($sub->end_date)->format('M d, Y'),
                'subscriptions_data' => $sub,
            ];
        });
    
        return response()->json([
            'status' => 'valid',
            'message' => 'Subscriptions fetched successfully.',
            'data' => $subscriptionsData
        ]);
    }


    
   
    
}
