<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Razorpay\Api\Api;
use App\Models\Subscription;
use Carbon\Carbon;
use Auth;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    // Generate order for frontend (App)
    public function createOrder(Request $request)
    {
        $request->validate([
            'plan_id'   => 'required|exists:plans,id',
            'course_id' => 'required'
        ]);
    
        $plan = \App\Models\Plan::find($request->plan_id);
        $userId = Auth::id();
    
        $api = new Api(
            config('services.razorpay.key'),
            config('services.razorpay.secret')
        );
    
        // Razorpay expects amount in paise
        $order = $api->order->create([
            'receipt'  => uniqid(),
            'amount'   => $plan->price * 100,
            'currency' => 'INR'
        ]);
    
        // Subscription dates (tentative, will confirm after payment)
        $startDate = Carbon::now();
        $endDate = $plan->billing_cycle === 'month'
            ? $startDate->copy()->addMonth()
            : $startDate->copy()->addYear();
    
        // Check if user already has expired subscription for this course
        $subscription = Subscription::where('user_id', $userId)
            ->where('course_id', $request->course_id)
            ->where('status', 'expired')
            ->latest()
            ->first();
        
        if ($subscription) {
            // Update old expired subscription to pending (renewal)
            $subscription->update([
                'plan_id'           => $plan->id,
                'start_date'        => $startDate,
                'end_date'          => $endDate,
                'status'            => 'pending',
                'razorpay_order_id' => $order['id'],
                'razorpay_payment_id' => null,
            ]);
        } else {
            // Create new subscription
            $subscription = Subscription::create([
                'user_id'           => $userId,
                'course_id'         => $request->course_id,
                'plan_id'           => $plan->id,
                'start_date'        => $startDate,
                'end_date'          => $endDate,
                'status'            => 'pending',
                'razorpay_order_id' => $order['id'],
            ]);
        }
    
        return response()->json([
            'status'          => 'valid',
            'order_id'        => $order['id'],
            'amount'          => $plan->price * 100,
            'currency'        => 'INR',
            'key'             => config('services.razorpay.key'),
            'plan'            => $plan,
            'subscription_id' => $subscription->id
        ]);
    }


    // Verify Payment and Activate Subscription
    public function verifyPayment(Request $request)
    {
        $request->validate([
            'razorpay_payment_id' => 'required',
            'razorpay_order_id'   => 'required',
            'razorpay_signature'  => 'required', // still needed for verification
            'subscription_id'     => 'required|exists:subscriptions,id'
        ]);
    
        $api = new Api(
            config('services.razorpay.key'),
            config('services.razorpay.secret')
        );
    
        $attributes = [
            'razorpay_order_id'   => $request->razorpay_order_id,
            'razorpay_payment_id' => $request->razorpay_payment_id,
            'razorpay_signature'  => $request->razorpay_signature
        ];
    
        try {
            $api->utility->verifyPaymentSignature($attributes);
    
            // Update subscription to active
            $subscription = Subscription::with(['course', 'plan'])->find($request->subscription_id);
            $subscription->update([
                'status'              => 'active',
                'razorpay_payment_id' => $request->razorpay_payment_id
                // 🔹 not saving razorpay_signature anymore
            ]);
            
            // Format response like subscription API
            $formattedSubscription = [
                'course_image' => $subscription->course ? asset('uploads/courses/' . $subscription->course->image) : null,
                'course_name'  => $subscription->course->name ?? null,
                'plan_name'    => $subscription->plan->name ?? null,
                'end_date'     => Carbon::parse($subscription->end_date)->format('M d, Y'),
                'subscriptions_data' => $subscription
            ];

    
            return response()->json([
                'status'  => 'valid',
                'message' => 'Payment successful & subscription activated',
                'subscription' => $formattedSubscription
            ]);
    
        } catch (\Exception $e) {
            // Mark as failed
            $subscription = Subscription::find($request->subscription_id);
            if ($subscription) {
                $subscription->update(['status' => 'failed']);
            }
    
            return response()->json([
                'status'  => 'error',
                'message' => 'Payment verification failed',
                'error'   => $e->getMessage()
            ], 400);
        }
    }
}
