<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Feedback;
use App\Models\SubChapterFavorite;
use Illuminate\Support\Facades\Validator;

class FeedbackController extends Controller
{
    public function store(Request $request)
    {
        $user = auth()->user();

        $validator = Validator::make($request->all(), [
            'feedback_for_type' => 'required|in:note,clinical_scenario,past_paper,reasoning_question',
            'feedback_for_id' => 'required|integer',
            'type' => 'required|in:correction,suggestion',
            'message' => 'required|string|max:2000',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'invalid',
                'message' => $validator->errors()->first()
            ]);
        }

         // Check if the feedback already exists
        $existingFeedback = Feedback::where('user_id', $user->id)
            ->where('feedback_for_type', $request->feedback_for_type)
            ->where('feedback_for_id', $request->feedback_for_id)
            ->where('type', $request->type)
            ->first();

        if ($existingFeedback) {
            return response()->json([
                'status' => 'exists',
                'message' => 'This feedback is already created.'
            ]);
        }

        $feedback = Feedback::create([
            'user_id' => $user->id,
            'feedback_for_type' => $request->feedback_for_type,
            'feedback_for_id' => $request->feedback_for_id,
            'type' => $request->type,
            'message' => $request->message,
        ]);

        return response()->json([
            'status' => 'valid',
            'message' => 'Feedback submitted successfully.',
            'data' => $feedback
        ]);
    }

    //  public function favorites()
    // {
    //     try {
    //         $user = auth()->user();
    //         $sub_chapters = SubChapterFavorite::with('subChapter.chapter')->select('id', 'sub_chapter_id','is_favorite')
    //             ->where('is_favorite', 1)
    //             ->where('user_id', $user->id)
    //             ->orderBy('created_at', 'asc')
    //             ->get();
                
            

    //         $data = [];
    //         foreach ($sub_chapters as $subChapter) {
                
    //             if (!$subChapter->subChapter) {
    //                 // Skip if relation is missing
    //                 continue;
    //             }
                
    //             $sub_chapter_array = $subChapter->subChapter->toArray();
    //             $sub_chapter_array['is_favorite'] = true;

    //              $data[] = [
    //                 'id'        => (int)$subChapter->sub_chapter_id,
    //                 'title'     => optional($subChapter->subChapter)->title,
    //                 'sub_title' => optional(optional($subChapter->subChapter)->chapter)->chapter_type_label 
    //                                 . ' - Chapter ' 
    //                                 . optional($subChapter->subChapter)->order_index,
    //                 //'sub_chapter' => $subChapter->subChapter->toArray(),
    //                 'sub_chapter' => $sub_chapter_array,
    //             ];

    //         }

    //         return response()->json([
    //             'status'  => 'valid',
    //             'message' => 'Favorites fetched successfully',
    //             'data'    => $data
    //         ]);

    //     } catch (\Exception $e) {
    //         return response()->json([
    //             'status'  => 'error',
    //             'message' => 'Something went wrong.',
    //             'error'   => $e->getMessage()
    //         ], 500);
    //     }
    // }
    
    public function favorites()
    {
        try {
            $user = auth()->user();
    
            // Fetch all favorited sub chapters with their related chapter and course
            $sub_chapters = SubChapterFavorite::with('subChapter.chapter')
                ->select('id', 'sub_chapter_id', 'is_favorite')
                ->where('is_favorite', 1)
                ->where('user_id', $user->id)
                ->orderBy('created_at', 'asc')
                ->get();
    
            $data = [];
    
            foreach ($sub_chapters as $subChapter) {
    
                if (!$subChapter->subChapter || !$subChapter->subChapter->chapter) {
                    // Skip if relation is missing
                    continue;
                }
    
                $subChapterModel = $subChapter->subChapter;
                $chapter = $subChapterModel->chapter;
    
                // Check if user is subscribed to the course
                $hasActiveSubscription = \App\Models\Subscription::where('user_id', $user->id)
                    ->where('course_id', $chapter->course_id)
                    ->where('status', 'active')
                    ->whereDate('end_date', '>=', now())
                    ->exists();
    
                // Convert sub_chapter to array
                $sub_chapter_array = $subChapterModel->toArray();
    
                // If not free and user is subscribed, override to free
                if ($subChapterModel->is_free == 0 && $hasActiveSubscription) {
                    $sub_chapter_array['is_free'] = true;
                }
    
                // Always mark as favorite
                $sub_chapter_array['is_favorite'] = true;
    
                $data[] = [
                    'id'        => (int)$subChapter->sub_chapter_id,
                    'title'     => $subChapterModel->title,
                    'sub_title' => optional($chapter)->chapter_type_label . ' - Chapter ' . $subChapterModel->order_index,
                    'sub_chapter' => $sub_chapter_array,
                ];
            }
    
            return response()->json([
                'status'  => 'valid',
                'message' => 'Favorites fetched successfully',
                'data'    => $data
            ]);
    
        } catch (\Exception $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Something went wrong.',
                'error'   => $e->getMessage()
            ], 500);
        }
    }


    public function favoriteCreate(Request $request)
    {
        $user = auth()->user();

        $validator = Validator::make($request->all(), [
            'sub_chapter_id' => 'required|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'invalid',
                'message' => $validator->errors()->first()
            ]);
        }

        // Check if the favorite already exists
        $existingFavorite = SubChapterFavorite::where('user_id', $user->id)
            ->where('sub_chapter_id', $request->sub_chapter_id)
            ->first();

        if ($existingFavorite) {
            $existingFavorite->delete();     
            return response()->json([
                'status' => 'valid',
                'message' => 'Favorite removed successfully'
            ]);
        }

        $fav = SubChapterFavorite::create([
            'user_id' => $user->id,
            'sub_chapter_id' => $request->sub_chapter_id,
        ]);      

        return response()->json([
            'status' => 'valid',
            'message' => 'Favorite created successfully.'
        ]);
    }

}
