<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Course;
use App\Models\Chapter;
use App\Models\SubChapter;
use App\Models\Content;
use App\Models\College;
use App\Models\CollegeContent;
use App\Models\AppUserContent;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class CourseController extends Controller
{

    public function index()
    {
        $courses = Course::select('id', 'name', 'image', 'is_active','priority', 'created_at', 'updated_at')
            ->orderBy('priority', 'asc')
            ->get();

        // Convert image path to full URL
        foreach ($courses as $course) {
            $course->image = $course->image ? asset('uploads/courses/' . $course->image) : null;
        }

        return response()->json([
            'status' => 'valid',
            'message' => 'Courses fetched successfully',
            'data' => $courses
        ]);
    }

    public function searchCourses(Request $request)
    {
        $keyword = $request->input('q'); // or use 'search' if frontend uses that key

        if (!$keyword) {
            return response()->json([
                'status' => 'invalid',
                'message' => 'Search keyword is required',
            ], 422);
        }

        $courses = Course::select('id', 'name', 'image', 'is_active', 'created_at', 'updated_at')
            ->where('is_active', 1)
            ->where('name', 'like', '%' . $keyword . '%')
            ->orderBy('created_at', 'asc')
            ->get();

        foreach ($courses as $course) {
            $course->image = $course->image ? asset('uploads/courses/' . $course->image) : null;
        }

        return response()->json([
            'status' => 'valid',
            'message' => 'Courses search result',
            'data' => $courses
        ]);
    }

    public function chapters(Request $request)
    {
        try {
            // Validate required fields
            $validator = Validator::make($request->all(), [
                'course_id'     => 'required|integer|exists:courses,id',
                'chapter_type'  => 'required|string|in:note,clinical_scenario,reasoning_question,mcq,neet_pg_mcq',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status'  => 'invalid',
                    'message' => 'Validation failed',
                    'errors'  => $validator->errors()
                ], 422);
            }

            // Fetch chapters
            $chapters = Chapter::select('id', 'title', 'order_index', 'chapter_content')
                ->where('course_id', $request->course_id)
                ->where('chapter_type', $request->chapter_type)
                ->orderBy('created_at', 'asc')
                ->get()
                ->map(function ($chapter) use ($request) {
                    // if ($request->chapter_type === 'reasoning_question') {
                    //     // Decode chapter_content JSON
                    //     $decoded = json_decode($chapter->chapter_content, true);
                    //     return [
                    //         'id' => $chapter->id,
                    //         'chapter_content' => $decoded ?? [],
                    //     ];
                    // } else {
                        // Default response
                        return [
                            'id' => $chapter->id,
                            'title' => $chapter->title,
                            'order_index' => $chapter->order_index,
                            'chapter' => $chapter->order_index ? 'Chapter ' . $chapter->order_index : null,
                        ];
                    //}
                });

            return response()->json([
                'status'  => 'valid',
                'message' => 'Chapters fetched successfully',
                'data'    => $chapters
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Something went wrong.',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function popularChapters()
    {
        try {
            // Get latest or highest order_index chapters (distinct by title)
            $popularChapters = Chapter::whereNotNull('title')
                ->whereNotNull('order_index')
                ->orderByDesc('order_index') // Or by created_at if you prefer
                //->whereNotNull('chapter_content')
                ->distinct('title')
                ->limit(10)
                ->get();

            foreach ($popularChapters as $chapter) {
                $chapter['sub_title'] = 'Chapter ' . $chapter->order_index;
            }

            return response()->json([
                'status'  => 'valid',
                'message' => 'Popular chapters fetched successfully',
                'data'    => $popularChapters
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Something went wrong.',
                'error'   => $e->getMessage()
            ], 500);
        }
    }


    public function subChapters(Request $request)
    {
        try {

            $user = auth()->user();
            // Validate required fields
            $validator = Validator::make($request->all(), [
                'chapter_id'    => 'required|integer|exists:chapters,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status'  => 'invalid',
                    'message' => 'Validation failed',
                    'errors'  => $validator->errors()
                ], 422);
            }

            // Fetch chapter with its course
            $chapter = Chapter::with('course')->findOrFail($request->chapter_id);

            // Check if user has active subscription for this course
            $hasActiveSubscription = \App\Models\Subscription::where('user_id', $user->id)
                ->where('course_id', $chapter->course_id)
                ->where('status', 'active')
                ->whereDate('end_date', '>=', now())
                ->exists();
                
             // Get all favorited sub_chapter_ids for the user
            $favoriteSubChapterIds = \App\Models\SubChapterFavorite::where('user_id', $user->id)
            ->pluck('sub_chapter_id')
            ->toArray();

            // Fetch chapters
            $sub_chapters = SubChapter::select('id', 'title', 'order_index','is_free','is_favorite')
                ->where('chapter_id', $request->chapter_id)
                ->orderBy('created_at', 'asc')
                ->get();

            // Append chapter label
            foreach ($sub_chapters as $chapter) {
                $chapter->chapter = $chapter->order_index ? 'Chapter ' . $chapter->order_index : null;

                if ($chapter->is_free == 0 && $hasActiveSubscription) {
                    $chapter->is_free = 1; // unlock paid chapter for subscribed users
                }
                
               //$chapter->is_free = $chapter->is_free == 1 ? false : true;
               // Set is_favorite = true/false based on existence
                $chapter->is_favorite = in_array($chapter->id, $favoriteSubChapterIds);

            }

            return response()->json([
                'status'  => 'valid',
                'message' => 'Sub Chapters fetched successfully',
                'data'    => $sub_chapters
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Something went wrong.',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function contents(Request $request)
    {
        try {
            // Validate required fields
            $validator = Validator::make($request->all(), [
                'sub_chapter_id' => 'required|integer|exists:sub_chapters,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status'  => 'invalid',
                    'message' => 'Validation failed',
                    'errors'  => $validator->errors()
                ], 422);
            }

            // Fetch and format contents
            $contents = Content::select('id', 'title', 'order_index', 'content_type', 'text_content', 'pdf_url')
                ->where('sub_chapter_id', $request->sub_chapter_id)
                ->orderBy('created_at', 'asc')
                ->get()
                ->map(function ($content) {
                    // Decode text_content if it’s valid JSON
                    $decoded = json_decode($content->text_content, true);
                    if ($decoded) {
                        // Remove <p> tags from question
                        if (isset($decoded['question'])) {
                            $decoded['question'] = strip_tags($decoded['question']);
                        }
                        
                        if (!empty($decoded['image'])) {
                            $decoded['image'] = asset('uploads/mcqs/' . $decoded['image']);
                        } else {
                            $decoded['image'] = null;
                        }
                        
                        $content->text_content = $decoded;
                    }else {
                        // If null or invalid JSON → set as blank string
                        $content->text_content = '';
                    }
                   if(in_array($content->content_type, ['note', 'clinical_scenario'])){
                        $content->pdf_url = $content->pdf_url ? asset('uploads/'.$content->content_type.'/' . $content->pdf_url) : null;
                    }else {
                        // If null or invalid JSON → set as blank string
                        $content->pdf_url = '';
                    }
                   
                    return $content;
                });

            return response()->json([
                'status'  => 'valid',
                'message' => 'Contents fetched successfully',
                'data'    => $contents
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Something went wrong.',
                'error'   => $e->getMessage()
            ], 500);
        }
    }
    
    public function submitContent(Request $request)
    {
        $user = auth()->user();

        $validator = Validator::make($request->all(), [
            'content_id' => 'required|integer',
            'user_answer' => 'required|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'invalid',
                'message' => $validator->errors()->first()
            ]);
        }

         // Check if the feedback already exists
        $existingContent = AppUserContent::where('user_id', $user->id)
            ->where('content_id', $request->content_id)
            ->first();

        if ($existingContent) {

            $existingContent->update([
                'user_answer' => $request->user_answer
            ]);

            return response()->json([
                'status' => 'exists',
                'message' => 'This answer is updated.',
            ]);
        }

        $appContent = AppUserContent::create([
            'user_id' => $user->id,
            'content_id' => $request->content_id,
            'user_answer' => $request->user_answer,
        ]);

        return response()->json([
            'status' => 'valid',
            'message' => 'Answer submitted successfully.',
            'data' => $appContent
        ]);
    }


    public function pastPapers()
    {
        $colleges = College::select('id', 'name')
            ->withCount('contents')
            ->orderBy('created_at', 'asc')
            ->get();

        // Convert image path to full URL
        foreach ($colleges as $college) {
            $college->sub_title = $college->contents_count . ' Pdf' . ($college->contents_count !== 1 ? 's' : '');
        }

        return response()->json([
            'status' => 'valid',
            'message' => 'Colleges fetched successfully',
            'data' => $colleges
        ]);
    }

    public function pastPapersDetail(Request $request)
    {
        try {
            // Validate required fields
            $validator = Validator::make($request->all(), [
                'college_id'     => 'required|integer|exists:colleges,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status'  => 'invalid',
                    'message' => 'Validation failed',
                    'errors'  => $validator->errors()
                ], 422);
            }

            $CollegeContent = CollegeContent::select('id', 'title', 'pdf_url')
                ->where('college_id', $request->college_id)
                ->orderBy('created_at', 'asc')
                ->get();

            // Append chapter label
            foreach ($CollegeContent as $chapter) {
                $chapter->pdf_url = $chapter->pdf_url ? asset('uploads/college/' . $chapter->pdf_url) : null;
            }

            return response()->json([
                'status'  => 'valid',
                'message' => 'Content fetched successfully',
                'data'    => $CollegeContent
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Something went wrong.',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

   
    
}
