<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\AppUser;
use App\Models\College;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Carbon\Carbon;

class AppAuthController extends Controller
{
    // 1. Login: Send OTP (only if user exists)
    public function sendOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'mobile_no' => 'required|digits:10'
        ]);
        if ($validator->fails()) {
            return response()->json([
                'status' => 'invalid',
                'message' => $validator->errors()->first()
            ]);
        }

        $user = AppUser::where('mobile_no', $request->mobile_no)->first();
        if (!$user) {
            return response()->json([
                'status' => 'invalid',
                'message' => 'User not found. Please register.'
            ]);
        }
        
        
        if (!$user->is_register && !$user->is_active) {
            return response()->json([
                'status' => 'invalid',
                'message' => 'User is not active. Please contact support.'
            ], 403);
        }
        
        

        $otp = rand(1000, 9999);
        $otp = 1234; // For testing, remove in production
        $expiresAt = Carbon::now()->addMinutes(10);

        $user->otp = $otp;
        $user->otp_expires_at = $expiresAt;
        $user->save();

        // TODO: Integrate SMS gateway here

        return response()->json([
            'status' => 'valid',
            'message' => 'OTP sent successfully'
        ]);
    }

    // 2. Register: Create user and send OTP
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string',
            'mobile_no' => 'required|digits:10|unique:app_users,mobile_no',
            'email' => 'nullable|email|unique:app_users,email',
            'college_id' => 'required|exists:colleges,id',
            'referral_code' => 'nullable|string|exists:app_users,referral_code',
            'profile_photo' => 'nullable|image|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'invalid',
                'message' => $validator->errors()->first()
            ]);
        }

        // Handle photo upload
        $photoPath = null;
        if ($request->hasFile('profile_photo')) {
            $photo = $request->file('profile_photo');
            $filename = uniqid() . '.' . $photo->getClientOriginalExtension();
            $photo->move(public_path('uploads'), $filename);
            $photoPath = $filename;
        }

        // If referral code is present, get referred_by_user_id
        $referredBy = null;
        if ($request->referral_code) {
            $refUser = AppUser::where('referral_code', $request->referral_code)->first();
            $referredBy = $refUser ? $refUser->id : null;
        }

        // Generate unique referral code for this user
        $myReferralCode = strtoupper(Str::random(8));

        // Generate OTP for verification
        $otp = rand(1000, 9999);
        $otp = 1234; // For testing, remove in production
        $expiresAt = Carbon::now()->addMinutes(10);

        $user = AppUser::create([
            'name' => $request->name,
            'mobile_no' => $request->mobile_no,
            'email' => $request->email,
            'college_id' => $request->college_id,
            'referral_code' => $myReferralCode,
            'referred_by_user_id' => $referredBy,
            'profile_photo_url' => $photoPath,
            'otp' => $otp,
            'otp_expires_at' => $expiresAt,
            'is_active' => 0, // Not active until OTP verified
        ]);

        // TODO: Integrate SMS gateway here

        return response()->json([
            'status' => 'valid',
            'message' => 'Registration successful. OTP sent for verification.',
            'data' => [
                'mobile_no' => $user->mobile_no
            ]
        ]);
    }

    // 3. Verify OTP (for both login and registration)
    public function verifyOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'mobile_no' => 'required|exists:app_users,mobile_no|digits:10',
            'otp' => 'required|digits:4'
        ]);
        if ($validator->fails()) {
            return response()->json([
                'status' => 'invalid',
                'message' => $validator->errors()->first()
            ]);
        }

        $user = AppUser::where('mobile_no', $request->mobile_no)
            ->where('otp', $request->otp)
            ->where('otp_expires_at', '>=', Carbon::now())
            ->first();

        if (!$user) {
            return response()->json([
                'status' => 'invalid',
                'message' => 'Invalid or expired OTP'
            ]);
        }

        // Clear OTP after successful verification
        $user->otp = null;
        $user->otp_expires_at = null;
        $user->is_active = 1; // Activate user if not already
        $user->is_register = 0; // User has completed registration
        $user->save();
        
        $user->profile_photo_url = $user->profile_photo_url ? asset('uploads/' . $user->profile_photo_url) : null;
        //$token = Str::random(60);
        $token = $user->createToken('mobile-app')->plainTextToken;

        return response()->json([
                'status' => 'valid',
                'message' => 'OTP verified',
                'data' => [
                'token' => $token,
                'user' => $user
            ]
        ]);

    }

    public function profileUpdate(Request $request)
    {
        $user = auth()->user(); // Authenticated user

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255',
            'email' => 'nullable|email|unique:app_users,email,' . $user->id,
            'college_id' => 'nullable|exists:colleges,id',
            'profile_photo' => 'nullable|image|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'invalid',
                'message' => $validator->errors()->first()
            ]);
        }

        if ($request->filled('name')) {
            $user->name = $request->name;
        }

        if ($request->filled('email')) {
            $user->email = $request->email;
        }

        if ($request->filled('college_id')) {
            $user->college_id = $request->college_id;
        }

        if ($request->hasFile('profile_photo')) {
            $photo = $request->file('profile_photo');
            $filename = uniqid() . '.' . $photo->getClientOriginalExtension();
            $photo->move(public_path('uploads'), $filename);
            $user->profile_photo_url = $filename;
        }

        $user->save();

        return response()->json([
            'status' => 'valid',
            'message' => 'Profile updated successfully',
            'data' => [
                'name' => $user->name,
                'mobile_no' => $user->mobile_no,
                'email' => $user->email,
                'college' => optional($user->college)->name,
                'referral_code' => $user->referral_code,
                'profile_photo_url' => $user->profile_photo_url 
                    ? asset('uploads/' . $user->profile_photo_url) 
                    : null
            ]
        ]);
    }


    // 4. College List
    public function colleges()
    {
        $colleges = College::select('id', 'name')->orderBy('name')->get();
        return response()->json([
            'status' => 'valid',
            'message' => 'College list fetched successfully',
            'data' => $colleges
        ]);
    }


    public function logout(Request $request)
    {
        $user = $request->user(); // Authenticated user

        // Revoke current access token
        $user->currentAccessToken()->delete();

        return response()->json([
            'status' => 'valid',
            'message' => 'Logged out successfully.'
        ]);
    }
    
    public function deleteAccount(Request $request)
    {
        $user = $request->user(); // Authenticated user

        try {
            \DB::transaction(function () use ($user) {
                // Delete user's favorites
                \App\Models\SubChapterFavorite::where('user_id', $user->id)->delete();

                // Delete subscriptions
               // \App\Models\Subscription::where('user_id', $user->id)->delete();

                // Delete profile photo if exists
                if ($user->profile_photo_url && file_exists(public_path('uploads/' . $user->profile_photo_url))) {
                    unlink(public_path('uploads/' . $user->profile_photo_url));
                }

                // Delete user account
                $user->delete();
            });

            return response()->json([
                'status' => 'valid',
                'message' => 'Account deleted successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'invalid',
                'message' => 'Failed to delete account. Please try again.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

}
