<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\SubChapter;
use App\Models\Chapter;
use App\Models\Course;

class SubChapterController extends Controller
{
    public function index()
    {
        //$sub_chapter = SubChapter::with(['chapter', 'contents'])->orderBy('created_at', 'asc')->get();
        $sub_chapter = SubChapter::with(['chapter.course', 'contents'])
            ->whereHas('chapter.course', function ($query) {
                $query->whereNull('deleted_at'); // Exclude soft-deleted courses
            })
            ->orderBy('created_at', 'asc')
        ->get();
        return view('admin.sub_chapters.index', compact('sub_chapter'));
    }

    public function create()
    {
        $courses = Course::where('is_active', true)->where('name', 'not like', '%Past Question Papers%')->orderBy('priority', 'asc')->get();
        $chapters = Chapter::get();
        return view('admin.sub_chapters.create', compact('courses','chapters'));
    }

    // Add this to your SubChapterController
    public function getChapters(Request $request)
    {
        $chapters = Chapter::where('course_id', $request->course_id)
                            ->where('chapter_type', $request->chapter_type)
                            ->get();

        return response()->json($chapters);
    }


    public function store(Request $request)
    {
        $data = $request->validate([
            'title' => 'nullable|string|max:255|unique:sub_chapters,title',
            'course_id' => 'required|exists:courses,id',
            'chapter_type' => 'nullable|string',
            'forced_chapter_type' => 'nullable|string',
            'chapter_id' => 'required|exists:chapters,id',
            'order_index' => 'required|integer',
            'is_free' => 'required',
        ]);
    
        // If forced value is passed, override
        if (!empty($data['forced_chapter_type'])) {
            $data['chapter_type'] = $data['forced_chapter_type'];
        }
        unset($data['forced_chapter_type']);
        
         $data['is_free'] = (int)$data['is_free'];
        $data['order_index'] = (int)$data['order_index'];
    
        SubChapter::create($data);
    
        notification(auth()->id(), $request->course_id);
    
        return redirect()->route('admin.sub_chapters.index')->with('success', 'Sub Chapter created successfully.');
    }


    public function edit(SubChapter $sub_chapter)
    {
        $courses = Course::where('is_active', true)->where('name', 'not like', '%Past Question Papers%')->orderBy('priority', 'asc')->get();
        $chapters = Chapter::get();
        return view('admin.sub_chapters.edit', compact('sub_chapter','courses','chapters'));
    }

    public function update(Request $request, SubChapter $sub_chapter)
    {
        $data = $request->validate([
            'title' => 'nullable|string|max:255|unique:sub_chapters,title,' . $sub_chapter->id,
            'course_id' => 'required|exists:courses,id',
            'chapter_type' => 'nullable|string',
            'forced_chapter_type' => 'nullable|string',
            'chapter_id' => 'required|exists:chapters,id',
            'order_index' => 'required|integer',
            'is_free' => 'required',
            //'is_favorite' => 'required', // optional
        ]);
    
        // If forced value is passed, override
        if (!empty($data['forced_chapter_type'])) {
            $data['chapter_type'] = $data['forced_chapter_type'];
        }
        unset($data['forced_chapter_type']);
    
        // Ensure proper type casting
        $data['is_free'] = (int)$data['is_free'];
        $data['order_index'] = (int)$data['order_index'];
    
        $sub_chapter->update($data);
    
        return redirect()->route('admin.sub_chapters.index')->with('success', 'Sub Chapter updated successfully.');
    }


    public function destroy(SubChapter $sub_chapter)
    {
        $sub_chapter->delete();
        return redirect()->route('admin.sub_chapters.index')->with('success', 'Sub Chapter deleted successfully.');
    }
}
