<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Content;
use App\Models\SubChapter;

class ContentController extends Controller
{

    public function entry($type, $sub_chapter_id)
    {
        if (in_array($type, ['note', 'clinical_scenario'])) {
            $content = Content::where('sub_chapter_id', $sub_chapter_id)->first();

            if (!$content) {
                $content = Content::create([
                    'sub_chapter_id' => $sub_chapter_id,
                    'chapter_type' => $type,
                    'title' => '',
                    'content_type' => $type, // important if you use this elsewhere
                    'order_index' => 1,
                ]);
            }

            if ($type === 'note') {
                return view('admin.contents.edit_note', compact('content', 'type'));
            }

            // Special case for 'clinical_scenario' — use its own view
            if ($type === 'clinical_scenario') {
                return view('admin.contents.edit_clinical_scenario', compact('content', 'type'));
            }

            
        }

        // Redirect to list view for MCQ, clinical_scenario, etc.
        // if ($type === 'mcq') {
        //     return redirect()->route('admin.contents.index', [
        //         'type' => $type,
        //         'sub_chapter_id' => $sub_chapter_id,
        //     ]);
        // }

        if (in_array($type, ['mcq', 'neet_pg_mcq', 'reasoning_question'])) {
            return redirect()->route('admin.contents.index', [
                'type' => $type,
                'sub_chapter_id' => $sub_chapter_id,
            ]);
        }

        abort(404); // fallback
    }



    public function index($type, $sub_chapter_id)
    {
        $contents = Content::where('sub_chapter_id', $sub_chapter_id)
            ->where('content_type', $type)
            ->get();

        return view('admin.contents.index', compact('contents', 'sub_chapter_id', 'type'));
    }


    public function create($type, $id)
    {
        $subChapter = SubChapter::findOrFail($id);
        return view('admin.contents.create', compact('type', 'subChapter'));
    }

    public function store(Request $request, $type, $id)
    {
        $subChapter = SubChapter::findOrFail($id);

        if (in_array($type, ['note','clinical_scenario'])) {
            
            $data = $request->validate([
                'title' => 'required|string|max:255',
                'file' => 'required|mimes:pdf|max:5120', // max 5MB
            ]);

            if ($request->hasFile('file')) {
                $pdf = $request->file('file');
                $filename = uniqid($type.'_') . '.' . $pdf->getClientOriginalExtension();
                $pdf->move(public_path('uploads/'.$type), $filename);
                $data['pdf_url'] = $filename;
            }


            Content::create([
                'sub_chapter_id' => $subChapter->id,
                'content_type' => $type,
                'title' => $data['title'],
                'pdf_url' => $data['pdf_url'],
                'order_index' => 1, // Adjust as needed
            ]);

        } elseif (in_array($type, ['mcq','neet_pg_mcq'])) {
            $data = $request->validate([
                'title' => 'required|string|max:255',
                'question' => 'required|string',
                'option_1' => 'required|string',
                'option_2' => 'required|string',
                'option_3' => 'required|string',
                'option_4' => 'required|string',
                'correct_option' => 'required|in:1,2,3,4',
                'explanation' => 'nullable|string',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            ]);

            $imageFilename = null;
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                //$imageFilename = uniqid('mcq_') . '.' . $image->getClientOriginalExtension();
                //$image->move(public_path('uploads/mcqs'), $imageFilename);
                
                $imageFilename = uniqid($type.'_') . '.' . $image->getClientOriginalExtension();
                $image->move(public_path('uploads/'.$type), $imageFilename);
            }

            $jsonContent = json_encode([
                'question' => $data['question'],
                'options' => [
                    $data['option_1'],
                    $data['option_2'],
                    $data['option_3'],
                    $data['option_4'],
                ],
                'correct' => $data['correct_option'],
                'explanation' => $data['explanation'],
                'image' => $imageFilename,
            ]);

            Content::create([
                'sub_chapter_id' => $subChapter->id,
                'content_type' => $type,
                'title' => $data['title'],
                'text_content' => $jsonContent,
                'order_index' => 1, // Adjust as needed
            ]);
        } elseif ($type === 'reasoning_question') {
            $data = $request->validate([
                'question'    => 'required|string',
                'answer'      => 'required|string',
                'explanation' => 'nullable|string',
            ]);

            $jsonContent = json_encode([
                'question'    => $data['question'],
                'answer'      => $data['answer'],
                'explanation' => $data['explanation'],
            ]);

            Content::create([
                'sub_chapter_id' => $subChapter->id,
                'content_type'   => $type,
                'title'          => $data['question'], // optional: using question as title
                'text_content'   => $jsonContent,
                'order_index'    => 1,
            ]);
        }

        //return redirect()->route('admin.sub_chapters.index')->with('success', 'Content saved successfully.');
        return redirect()->route('admin.contents.index', [$type, $subChapter->id])
                     ->with('success', 'Content created successfully.');
    }

    public function edit(Content $content)
    {
        $type = $content->content_type;

        // Special case for 'note' — use its own view
        if ($type === 'note') {
            return view('admin.contents.edit_note', compact('content'));
        }

         // Special case for 'clinical_scenario' — use its own view
        if ($type === 'clinical_scenario') {
            return view('admin.contents.edit_clinical_scenario', compact('content'));
        }
        

        return view('admin.contents.edit', compact('content', 'type'));
    }

    public function update(Request $request, Content $content)
    {
        $type = $content->content_type;

        if (in_array($type, ['note','clinical_scenario'])) {

            $data = $request->validate([
                'title' => 'required|string|max:255',
                'file' => 'nullable|mimes:pdf|max:5120',
            ]);

            if ($request->hasFile('file')) {
                if ($content->pdf_url && file_exists(public_path('uploads/'.$type.'/' . $content->pdf_url))) {
                    unlink(public_path('uploads/'.$type.'/' . $content->pdf_url));
                }

                $pdf = $request->file('file');
                $filename = uniqid($type.'_') . '.' . $pdf->getClientOriginalExtension();
                $pdf->move(public_path('uploads/'.$type), $filename);
                $data['pdf_url'] = $filename;
            }

            $content->update($data);

       } elseif (in_array($type, ['mcq','neet_pg_mcq'])) {
            $data = $request->validate([
                'title' => 'required|string|max:255',
                'question' => 'required|string',
                'option_1' => 'required|string',
                'option_2' => 'required|string',
                'option_3' => 'required|string',
                'option_4' => 'required|string',
                'correct_option' => 'required|in:1,2,3,4',
                'explanation' => 'nullable|string',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            ]);

            $imageFilename = $json['image'] ?? null;

            if ($request->hasFile('image')) {
                if ($imageFilename && file_exists(public_path('uploads/mcqs/' . $imageFilename))) {
                    unlink(public_path('uploads/mcqs/' . $imageFilename));
                }

                $image = $request->file('image');
                $imageFilename = uniqid('mcq_') . '.' . $image->getClientOriginalExtension();
                $image->move(public_path('uploads/mcqs'), $imageFilename);
            }

            $jsonContent = json_encode([
                'question' => $data['question'],
                'options' => [
                    $data['option_1'],
                    $data['option_2'],
                    $data['option_3'],
                    $data['option_4'],
                ],
                'correct' => $data['correct_option'],
                'explanation' => $data['explanation'],
                'image' => $imageFilename,
            ]);

            $content->update([
                'title' => $data['title'],
                'text_content' => $jsonContent,
            ]);
        } elseif ($type === 'reasoning_question') {
            $data = $request->validate([
                'question'    => 'required|string',
                'answer'      => 'required|string',
                'explanation' => 'nullable|string',
            ]);

            $jsonContent = json_encode([
                'question'    => $data['question'],
                'answer'      => $data['answer'],
                'explanation' => $data['explanation'],
            ]);

            $content->update([
                'title'        => $data['question'], // optional
                'text_content' => $jsonContent,
            ]);
        }
        
        return redirect()->route('admin.sub_chapters.index')->with('success', 'Content updated successfully.');
        //return redirect()->back()->with('success', 'Content updated successfully.');
    }

    public function destroy(Content $content)
    {
        $type = $content->content_type;
        if (in_array($type, ['note','clinical_scenario']) && $content->image_url) {
            $path = public_path('uploads/'.$type.'/' . $content->image_url);
            if (file_exists($path)) {
                unlink($path);
            }
        }

        $content->delete();

        return redirect()->back()->with('success', 'Content deleted successfully.');
    }


}
